﻿using log4net;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.IWS.BatchService.Interface;
using VA.PPMS.IWS.BlobService.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.IWS.MappingService.Interface;
using VA.PPMS.IWS.PersistenceService.Interface;
using VA.PPMS.IWS.ProviderService.Interface;
using VA.PPMS.IWS.SchemaValidationService.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.ProviderService
{
    public class ProviderService : IProviderService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IBatchService _batchService;
        private readonly IBlobService _blobService;
        private readonly ISchemaValidationService _schemaValidationService;
        private readonly IMappingService _mappingService;
        private readonly IPersistenceService _persistenceService;

        public ProviderService(ILog logger, IIwsConfiguration configuration, IBatchService batchService, IBlobService blobService, ISchemaValidationService schemaValidationService, IMappingService mappingService, IPersistenceService persistenceService)
        {
            _logger = logger;
            _configuration = configuration;
            _batchService = batchService;
            _blobService = blobService;
            _schemaValidationService = schemaValidationService;
            _mappingService = mappingService;
            _persistenceService = persistenceService;
        }

        public async Task ProcessProvider(string message)
        {
            try
            {
                var dasMessage = new DasMessage(message);

                if (!dasMessage.IsValid) throw new PpmsServiceException("CCN Data Request - Unable to process request, data is invalid");

                var packet = await _configuration.GetCcnInitialDataRequestUrlAsync();
                dasMessage.Content = string.Format(packet.UrlPattern, packet.Host, packet.Port, dasMessage.Content);

                if (await CallUrl(dasMessage))
                {
                    await _batchService.UpdateBatch(dasMessage, "CCN Initial Data Request processed.", (int)ppms_batch_StatusCode.DataRequested);
                }
                else
                {
                    await _batchService.UpdateBatch(dasMessage, "CCN Initial Data Request failed GET request.", (int)ppms_batch_StatusCode.Failed);
                }
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem processing Provider: {message} @@@@", ex);
                throw;
            }
        }

        public async Task ProcessProviderPayload(string dasMessage)
        {
            try
            {
                var message = new DasMessage(dasMessage);
                var transactionId = message.TransactionId;

                _logger.Info($">>> INFO - Start Orchestration Processing for TransactionId: {transactionId} <<<");

                var providers = await _blobService.GetBlobAsync(dasMessage);

                if (providers == null)
                {
                    _logger.Error($"@@@@ ERROR - Unable to get blob for TransactionId: {transactionId}.");
                    return;
                }

                providers.IsVaNetwork = message.IsVaNetwork;
                var validationResult = await _schemaValidationService.ValidateSchemaAsync(providers);

                var validationSuccesses = validationResult.Where(x => string.IsNullOrEmpty(x.Result)).ToList();
                var result = providers.Provider.Join(validationSuccesses,
                        provider => provider.ProviderId,
                        successes => successes.ProviderId,
                        (provider, successes) => provider)
                    .ToList();

                var newProviders = new Providers
                {
                    NetworkId = message.SenderId,
                    TransactionId = message.TransactionId,
                    Provider = result,
                    IsVaNetwork = providers.IsVaNetwork
                };

                var batchList = new List<Validation>();
                var mapResults = new MapperResult();

                if (newProviders.Provider.Any())
                {
                    var size = newProviders.Provider.Count;
                    var i = 1;
                    MapperResult results = null;

                    foreach (var item in newProviders.Provider)
                    {
                        try
                        {
                            _logger.Info($"---- DEBUG - Orchestration Processing {i++} of {size}. [TransactionId: {transactionId}; ProviderId: {item.ProviderId}] ----");

                            var list = new Providers
                            {
                                TransactionId = message.TransactionId,
                                NetworkId = message.SenderId,
                                Provider = new List<Provider> { item },
                                IsVaNetwork = newProviders.IsVaNetwork
                            };

                            results = await _mappingService.MapAsync(list);
                        }
                        catch (Exception ex)
                        {
                            var msg = $"An error occurred trying to map provider info. Error: { ex.Message}";
                            if (results == null)
                            {
                                results = new MapperResult(msg);
                            }
                            else
                            {
                                results.Details[0].ValidationMessage = msg;
                            }

                            _logger.Error($"@@@@ ERROR - Unable to map provider info for TransactionId: {transactionId}. Message: {ex.Message} @@@@");
                        }

                        if (results != null)
                        {
                            mapResults.Add(results.Details);
                        }
                    }
                }

                _logger.Info($"---- INFO - Saving validation results for TransactionId: {transactionId} ----");

                UpdateValidations(validationResult, mapResults.Details);

                foreach (var validation in validationResult)
                {
                    try
                    {
                        var account = mapResults.FindProvider(validation.ProviderId);
                        if (account != null && string.IsNullOrEmpty(validation.CorrelationId))
                        {
                            validation.CorrelationId = account.Id.ToString();
                        }
                        batchList.Add(validation);
                    }
                    catch (Exception ex)
                    {
                        _logger.Error($"@@@@ ERROR - Unable to save batch info for TransactionId: {transactionId}. Message: {ex.Message} @@@@");
                    }
                }

                if (batchList.Any())
                {
                    _logger.Info($"@@@@ INFO - Saving batch info for TransactionId: {transactionId} @@@@");
                    await _persistenceService.SaveBatchAsync(batchList, message);
                }

                _logger.Info($">>> INFO - End Orchestration Processing for TransactionId: {transactionId} <<<");

                _logger.Info($"End ProviderPayloadQueueTrigger function processed: {dasMessage}");
            }
            catch (Exception e)
            {
                var message = dasMessage ?? "Undefined";
                _logger.Error($"!!!! ERROR - There was a problem with Orchestration Processing for TransactionId: {message}. The error is {e} !!!!");
                throw;
            }
        }

        private static async Task<bool> CallUrl(DasMessage message)
        {
            var url = message.Content;
            if (string.IsNullOrEmpty(url)) throw new PpmsServiceException("CCN Data Request - Unable to process request, URL is invalid");

            using (var client = new HttpClient())
            {
                client.BaseAddress = new Uri(url);
                client.DefaultRequestHeaders.Accept.Clear();
                //client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/xml"));
                client.DefaultRequestHeaders.Add("X-ConversationID", message.ConversationId);
                client.DefaultRequestHeaders.Add("X-RoutingSenderID", message.SenderId);
                client.DefaultRequestHeaders.Add("X-RoutingReceiverIDs", message.ReceiverId);
                client.DefaultRequestHeaders.Add("X-TransactionID", message.TransactionId);

                var response = await client.GetAsync(string.Empty);

                // TODO - Complete the implementation. Not sure what to do with the result
                return response.IsSuccessStatusCode;
            }
        }

        private static void UpdateValidations(List<Validation> validations, IList<MapperResultDetail> details)
        {
            foreach (var detail in details)
            {
                if (detail.IsValid) continue;

                var validation = validations.FirstOrDefault(x => x.ProviderId == detail.ProviderId);
                if (validation != null)
                {
                    validation.Result = detail.ValidationMessage;
                }
            }
        }
    }
}